@extends('adminlte::page')
@section('title', 'Ventas')

<head>
    <link rel="icon" href="{{ asset('icons/venta.png') }}" type="image/x-icon">
</head>
@section('content')
<div class="container mt-4">
    <div class="card shadow-lg p-4">
        <h2 class="text-center text-primary mb-4">📦 Realizar Venta</h2>

        <!-- Selección de Producto -->
        <div class="row">
            <div class="col-md-6">
                <label for="producto_id" class="form-label fw-bold">📌 Producto</label>
                <select id="producto_id" class="form-select">
                    <option value="">Seleccionar producto</option>
                    @foreach ($productos as $producto)
                        <option value="{{ $producto->id }}" data-nombre="{{ $producto->nombre }}" data-precio="{{ $producto->precio }}">
                            {{ $producto->nombre }} - S/ {{ number_format($producto->precio, 2) }}
                        </option>
                    @endforeach
                </select>
            </div>

            <div class="col-md-3">
                <label for="cantidad" class="form-label fw-bold">🔢 Cantidad</label>
                <input type="number" id="cantidad" class="form-control" min="1" value="1">
            </div>

            <div class="col-md-3 d-flex align-items-end">
                <button id="agregar_producto" class="btn btn-primary w-100">
                    ➕ Agregar Producto
                </button>
            </div>
        </div>
    </div>

    <!-- Tabla de Productos -->
    <div class="card shadow-lg p-4 mt-4">
        <h3 class="text-center text-success mb-3">🛒 Productos en la Venta</h3>
        <div class="table-responsive">
            <table class="table table-hover text-center">
                <thead class="table-dark">
                    <tr>
                        <th>Producto</th>
                        <th>Cantidad</th>
                        <th>Precio Unitario</th>
                        <th>Subtotal</th>
                        <th>Acciones</th>
                    </tr>
                </thead>
                <tbody id="tabla_ventas">
                    <!-- Productos agregados aparecerán aquí -->
                </tbody>
            </table>
        </div>
        <h4 class="text-end text-danger mt-3 fw-bold">
            Total: S/ <span id="total_venta">0.00</span>
        </h4>
    </div>

    <!-- Sección de Pago -->
    <div class="card shadow-lg p-4 mt-4">
        <h3 class="text-center text-info mb-3">💰 Pago</h3>
        <div class="row">
            <div class="col-md-6">
                <label for="monto_pagado" class="form-label fw-bold">💵 Monto Pagado</label>
                <input type="number" id="monto_pagado" class="form-control" min="0">
            </div>

            <div class="col-md-6">
                <h4 class="text-end mt-4">
                    🔄 Vuelto: S/ <span id="vuelto">0.00</span>
                </h4>
            </div>
        </div>
    </div>

    <button id="realizar_venta" class="btn btn-success mt-4 w-100 fs-5 fw-bold">
        ✅ Registrar Venta
    </button>
</div>


<script>
document.addEventListener('DOMContentLoaded', function () {
    let productosVenta = [];
    let totalVenta = 0;
    let csrfToken = '{{ csrf_token() }}';
    let urlVenta = "{{ route('ventas.store') }}";

    function actualizarTotal() {
        totalVenta = productosVenta.reduce((sum, p) => sum + p.subtotal, 0);
        document.getElementById('total_venta').textContent = totalVenta.toFixed(2);
        calcularVuelto();
    }

    function calcularVuelto() {
        let montoPagado = parseFloat(document.getElementById('monto_pagado').value) || 0;
        let vuelto = montoPagado - totalVenta;
        document.getElementById('vuelto').textContent = vuelto >= 0 ? vuelto.toFixed(2) : 'Monto insuficiente';
    }

    document.getElementById('agregar_producto').addEventListener('click', function () {
        let select = document.getElementById('producto_id');
        let cantidad = parseInt(document.getElementById('cantidad').value) || 1;
        let productoId = select.value;
        let nombre = select.options[select.selectedIndex].getAttribute('data-nombre');
        let precio = parseFloat(select.options[select.selectedIndex].getAttribute('data-precio')) || 0;

        if (!productoId) return alert('Selecciona un producto');

        let productoExistente = productosVenta.find(p => p.id == productoId);
        if (productoExistente) {
            productoExistente.cantidad += cantidad;
            productoExistente.subtotal = productoExistente.cantidad * productoExistente.precio;
        } else {
            productosVenta.push({ id: productoId, nombre, cantidad, precio, subtotal: cantidad * precio });
        }

        renderTabla();
        actualizarTotal();
    });

    function renderTabla() {
        let tbody = document.getElementById('tabla_ventas');
        tbody.innerHTML = '';

        productosVenta.forEach((p, index) => {
            let row = `<tr>
                <td>${p.nombre}</td>
                <td>${p.cantidad}</td>
                <td>S/ ${p.precio.toFixed(2)}</td>
                <td>S/ ${p.subtotal.toFixed(2)}</td>
                <td><button class="btn btn-danger btn-sm" onclick="eliminarProducto(${index})">Eliminar</button></td>
            </tr>`;
            tbody.innerHTML += row;
        });
    }

    window.eliminarProducto = function (index) {
        productosVenta.splice(index, 1);
        renderTabla();
        actualizarTotal();
    };

    document.getElementById('monto_pagado').addEventListener('input', calcularVuelto);

    document.getElementById('realizar_venta').addEventListener('click', function () {
        if (productosVenta.length === 0) {
            alert('Agrega productos antes de realizar la venta.');
            return;
        }

        let montoPagado = parseFloat(document.getElementById('monto_pagado').value) || 0;
        if (montoPagado < totalVenta) {
            alert('El monto pagado es insuficiente.');
            return;
        }

        fetch(urlVenta, {
            method: 'POST',
            headers: { 
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': csrfToken
            },
            credentials: 'same-origin',
            body: JSON.stringify({ productos: productosVenta, monto_pagado: montoPagado })
        }).then(response => response.json())
          .then(data => {
              if (data.success) {
                  alert('Venta realizada con éxito.');
                  productosVenta = []; 
                  renderTabla(); 
                  actualizarTotal(); 
                  document.getElementById('monto_pagado').value = ''; 
                  document.getElementById('vuelto').textContent = '0.00';
              } else {
                  alert('Error: ' + data.error);
              }
          }).catch(error => console.error('Error:', error));
    });
});
</script>
@endsection
