<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Http\Requests\Auth\LoginRequest;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\View\View;
use Illuminate\Support\Facades\RateLimiter;

class AuthenticatedSessionController extends Controller
{
    /**
     * Muestra la vista de login.
     */
    public function create(): View
    {
        return view('auth.login');
    }

    /**
     * Maneja la autenticación del usuario.
     */
    public function store(Request $request)
{
    // Limitar los intentos de inicio de sesión
    $key = 'login:' . $request->ip();  // Usamos la IP del usuario para la limitación
    if (RateLimiter::remaining($key, 3) === 0) {
        return back()->withErrors(['email' => 'Has superado el número de intentos. Por favor, espera unos minutos.']);
    }

    $credentials = $request->only('email', 'password');

    // Buscar al usuario por correo
    $user = \App\Models\User::where('email', $credentials['email'])->first();

    if (!$user) {
        RateLimiter::hit($key, 60);
        return back()->withErrors(['email' => 'Las credenciales no son correctas.']);
    }

    // Verificar si el usuario está inactivo
    if (!$user->activo) {
        return back()->withErrors(['email' => 'Tu cuenta está inactiva. Contacta al administrador.']);
    }

    // Intentar autenticar al usuario
    if (Auth::attempt($credentials)) {
        $request->session()->regenerate();

        // Redirigir a la página de 'dashboard' después de iniciar sesión
        return redirect()->route('dashboard');
    }

    // Incrementar el contador de intentos
    RateLimiter::hit($key, 60);  // Limitar a 3 intentos por minuto

    return back()->withErrors([
        'email' => 'Las credenciales no son correctas.',
    ]);
}


    /**
     * Cierra la sesión del usuario autenticado.
     */
    public function destroy(Request $request)
    {
        Auth::guard('web')->logout();
        $request->session()->invalidate();
        $request->session()->regenerateToken();

        return redirect('/login');
    }
    
}
